#!/bin/sh
#
# Set file permissions of Grid Engine distribution
#
# (c) 2002 Sun Microsystems, Inc. Use is subject to license terms.  
#
#
# The $OPTFILES are not mandatory for a distribution and will be set only if
# they exist
#
# This script must be called by user root on a machine where user root has
# permissions to change the ownership of a file
# 
# It is not necessary to run this script if the distribtuon has been
# installed with pkgadd, since pkgadd takes care about the correct
# permissions.
#

PATH=/bin:/usr/bin:/usr/bsd:/usr/sbin

SECFILELIST="bin lib utilbin"

FILELIST="3rd_party bin ckpt examples install_execd install_qmaster mpi \
          pvm qmon util utilbin"

OPTFILES="catman doc locale man inst_sge inst_sgeee" 

CLIENTFILES="qacct qalter qconf qdel qhost qlogin qmod qmon qrsh qsh \
             qstat qsub"

umask 022

#---------------------------------------------------
# SetFilePerm
#
SetFilePerm()
{
   f="$1"
   user="$2"
   group="$3"

   $INFOTEXT "Verifying and setting file permissions and owner in >%s<" $f

   chmod -R go+r $f
   find $f -type d -exec chmod 755 {} \;
   find $f -type f -perm -100 -exec chmod go+x {} \;
   chown -R $user $f
   chgrp -R $group $f
}

#--------------------------------------------------------------------------
# THE MAIN PROCEDURE
#--------------------------------------------------------------------------

instauto=false
instresport=false

if [ -z "$SGE_ROOT" -o ! -d "$SGE_ROOT" ]; then
   echo 
   echo ERROR: Please set your \$SGE_ROOT environment variable
   echo and restart this script. Exit.
   echo 
   exit 1
fi

if [ ! -f "$SGE_ROOT/util/arch" ]; then
   echo 
   echo ERROR: The shell script \"$SGE_ROOT/util/arch\" does not exist.
   echo Please verify your distribution and restart this script. Exit.
   echo
   exit 1
fi

if [ ! -f $SGE_ROOT/util/arch_variables ]; then
   echo
   echo ERROR: Missing shell script \"$SGE_ROOT/util/arch_variables\".
   echo Please verify your distribution and restart this script. Exit.
   echo
   exit 1
fi

. $SGE_ROOT/util/arch_variables

#---------------------------------------
# setup INFOTEXT begin
#---------------------------------------

V5BIN=$SGE_ROOT/bin/$ARCH
V5UTILBIN=$SGE_ROOT/utilbin/$ARCH       
# INFOTXT_DUMMY is needed by message parsing script
# which is looking for $INFOTEXT and would report
# errors in the next command. Please use INFOTXT_DUMMY
# instead of using $INFOTEXT

INFOTXT_DUMMY=$V5UTILBIN/infotext
INFOTEXT=$INFOTXT_DUMMY
if [ ! -x $INFOTXT_DUMMY ]; then
   echo "can't find binary \"$INFOTXT_DUMMY\""
   echo "Installation failed."
   exit 1
fi
SGE_INFOTEXT_MAX_COLUMN=5000; export SGE_INFOTEXT_MAX_COLUMN

#---------------------------------------
# setup INFOTEXT end
#---------------------------------------

if [ $# -lt 3 ]; then
   $INFOTEXT -e "\nSet file permissions and owner of Grid Engine distribution in \$SGE_ROOT\n\n" \
                "Usage: %s [-auto] [-resport] \"adminuser\" \"group\" <sge_root>\n\n" \
                "Example:\n\n" \
                "   # %s sgeadmin adm \$SGE_ROOT\n" `basename $0` `basename $0`
   exit 1
fi

if [ $1 = -auto ]; then
   instauto=true
   shift
fi

if [ $1 = -resport ]; then
   instresport=true
   shift
elif [ $1 = -noresport ]; then
   instresport=false
   shift
fi

if [ $3 = / -o $3 = /etc ]; then
   $INFOTEXT -e "\nERROR: cannot set permissions in \"%s\" directory of your system\n" $3
   exit 1
fi

if [ `echo $3 | env LC_ALL=C cut -c1` != / ]; then
   $INFOTEXT -e "\nERROR: please provide an absolute path for the distribution\n"
   exit 1
fi


if [ $instauto = true ]; then
   :
else
   clear
   $INFOTEXT "\n                    WARNING WARNING WARNING\n" \
             "                    -----------------------\n\n" \
             "We will set the the file ownership and permission to\n\n" \
             "   User:         %s\n" \
             "   Group:        %s\n" \
             "   In directory: %s\n\n" \
             "We will also install the following binaries as SUID-root:\n\n" \
             "   \$SGE_ROOT/utilbin/<arch>/rlogin\n" \
             "   \$SGE_ROOT/utilbin/<arch>/rsh\n" \
             "   \$SGE_ROOT/utilbin/<arch>/testsuidroot\n" $1 $2 $3

   $INFOTEXT -n -ask "yes" "no" \
            "Do you want to set the file permissions (yes/no) [no] >> "

   if [ $? = 1 ]; then
      $INFOTEXT "We will not set the file permissions. Exit."
      exit 1
   fi
fi

cd $3
if [ $? != 0 ]; then
   $INFOTEXT -e "ERROR: can't change to directory \"%s\". Exit." $3
   exit 1
fi

for f in $FILELIST $OPTFILES; do
   if [ -d $f -o -f $f ]; then
      SetFilePerm $f $1 $2
   fi
done

# These files and dirs are owned by root for security reasons
for f in $SECFILELIST; do
   if [ -d $f -o -f $f ]; then
      SetFilePerm $f root $ROOTGROUP
   fi
done

# Set permissions of SGE_ROOT itself
chown $1 .
chgrp $2 .
chmod 755 .

chown 0 utilbin/*/rsh utilbin/*/rlogin utilbin/*/testsuidroot
chgrp 0 utilbin/*/rsh utilbin/*/rlogin utilbin/*/testsuidroot
chmod 4511 utilbin/*/rsh utilbin/*/rlogin utilbin/*/testsuidroot

if [ $instresport = true ]; then
   for i in $CLIENTFILES; do
      chown 0 bin/*/$i
      chmod 4511 bin/*/$i   
   done
fi

$INFOTEXT "\nYour file permissions were set\n"
