#' Chart of columns correlation with a selected dimension
#'
#' This function allows to calculate the correlation (sqrt(COS2)) of the column categories with the
#' selected dimension.
#'
#' The function displays the correlation of the column categories with the selected dimension; the
#' parameter categ.sort=TRUE arrange the categories in decreasing order of correlation. At the
#' left-hand side, the categories' labels show a symbol (+ or -) according to which side of the
#' selected dimension they are correlated, either positive or negative. The categories are grouped
#' into two groups: categories correlated with the positive ('pole +') or negative ('pole -') pole
#' of the selected dimension. At the right-hand side, a legend (which is enabled/disabled using the
#' 'leg' parameter) indicates the row categories' contribution (in permills) to the selected
#' dimension (value enclosed within round brackets), and a symbol (+ or -) indicating whether they
#' are actually contributing to the definition of the positive or negative side of the dimension,
#' respectively. Further, an asterisk (*) flags the categories which can be considered major
#' contributors to the definition of the dimension.
#' @param data Name of the dataset (must be in dataframe format).
#' @param x Dimension for which the column categories correlation is returned (1st dimension by
#'   default).
#' @param categ.sort Logical value (TRUE/FALSE) which allows to sort the categories in descending
#'   order of correlation with the selected dimension. TRUE is set by default.
#' @param filter Filter the row categories listed in the top-right legend, only showing those who
#'   have a major contribution to the definition of the selected dimension.
#' @param leg Enable (TRUE; default) or disable (FALSE) the legend at the right-hand side of the
#'   dot plot.
#' @param dotprightm Increases the empty space between the right margin of the dot plot and the left
#'   margin of the legend box.
#' @param cex.leg Adjust the size of the legend's characters.
#' @param cex.labls Adjust the size of the dot plot's labels.
#' @param leg.x.spc Adjust the horizontal space of the chart's legend. See more info from the
#'   'legend' function's help (?legend).
#' @param leg.y.spc Adjust the y interspace of the chart's legend. See more info from the 'legend'
#'   function's help (?legend).
#' @keywords cols.corr
#' @export
#' @examples
#' data(greenacre_data)
#'
#' #Plots the correlation of the column categories with the 1st CA dimension.
#' cols.corr(greenacre_data, 1, categ.sort=TRUE)
#'
#' @seealso \code{\link{cols.corr.scatter}} , \code{\link{rows.corr}} ,
#'   \code{\link{rows.corr.scatter}}
#'   
cols.corr <- function (data, x = 1, categ.sort = TRUE, filter= FALSE, leg=TRUE, dotprightm=5, cex.leg=0.6, cex.labls=0.75, leg.x.spc=1, leg.y.spc=1) {
  
  cntr=NULL
  
  cadataframe <- CA(data, graph = FALSE)
  df <- data.frame(corr = round(sqrt((cadataframe$col$cos2[, x])), digits = 3), coord=cadataframe$col$coord[,x])
  df$labels <- ifelse(df$coord < 0,
                      paste(rownames(df), " - ", sep = ""), 
                      paste(rownames(df), " + ", sep = ""))
  df.row.cntr <- data.frame(coord=cadataframe$row$coord[,x], cntr=(cadataframe$row$contrib[,x]*10))
  df.row.cntr$labels <- ifelse(df.row.cntr$coord < 0,
                               paste(rownames(df.row.cntr), " - ", sep = ""), 
                               paste(rownames(df.row.cntr), " + ", sep = ""))
  df.row.cntr$specif <- ifelse(df.row.cntr$cntr > (100/nrow(data)) * 10, 
                               "*", 
                               "")
  df.row.cntr$specif2 <- paste0(df.row.cntr$specif, df.row.cntr$labels, "(", round(df.row.cntr$cntr,2), ")")
  ifelse(categ.sort == TRUE, 
         df.to.use <- df[order(-df$corr), ], 
         df.to.use <- df)
  df.to.use$pole <- ifelse(df.to.use$coord > 0, 
                           "pole +", 
                           "pole -")
  ifelse(filter== FALSE, 
         df.row.cntr <- df.row.cntr, 
         df.row.cntr <- subset(df.row.cntr, cntr>(100/nrow(data))*10))
  if(leg==TRUE){ 
    par(oma=c(0,0,0,dotprightm))
  } else {}
  dotchart2(df.to.use$corr, 
            labels = df.to.use$labels,
            groups=df.to.use$pole,
            sort. = FALSE,
            lty = 2, 
            xlim = c(0, 1), 
            cex.labels=cex.labls, 
            xlab = paste("Column categories' correlation with Dim. ", x))
  par(oma=c(0,0,0,0))
  if(leg==TRUE){ 
  legend(x="topright", 
         legend=df.row.cntr[order(-df.row.cntr$cntr),]$specif2, 
         xpd=TRUE, 
         cex=cex.leg, 
         x.intersp = leg.x.spc, 
         y.intersp = leg.y.spc)
  } else {}
}